<?php
// panel/ortam-kutuphanesi.php
require_once __DIR__ . '/init.php';

require_login();
auth_refresh_user();

$u = current_user();
$isAdmin = (($u['rol'] ?? '') === 'admin');
$tesisId = (int)($u['id'] ?? 0);

const MAX_UPLOAD_BYTES = 10485760; // 10MB
const MAX_BULK_DELETE  = 200;
const ALLOWED_MIMES = ['image/jpeg', 'image/png', 'image/x-png', 'image/gif', 'image/webp'];
const MAX_IMAGE_HEIGHT = 1024;
function json_out(array $data, int $code = 200): never {
  if (!headers_sent()) {
    http_response_code($code);
    header('Content-Type: application/json; charset=utf-8');
  }
  echo json_encode($data, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
  exit;
}

function allowed_mime(string $mime): bool {
  return in_array($mime, ALLOWED_MIMES, true);
}

function random_code(int $len = 10): string {
  $chars = 'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789';
  $out = '';
  $max = strlen($chars) - 1;
  for ($i = 0; $i < $len; $i++) $out .= $chars[random_int(0, $max)];
  return $out;
}

function uploads_rel_dir(int $tesisId): string {
  $y = date('Y');
  $m = date('m');
  return "uploads/medya/{$tesisId}/{$y}/{$m}";
}

function uploads_abs_dir(int $tesisId): string {
  $rel = uploads_rel_dir($tesisId);
  $abs = rtrim(ROOT_PATH, '/\\') . '/' . $rel;
  if (!is_dir($abs)) {
    if (!@mkdir($abs, 0755, true) && !is_dir($abs)) {
      throw new RuntimeException('Yükleme klasörü oluşturulamadı.');
    }
  }
  return $abs;
}

function abs_path_from_rel(string $rel): string {
  return rtrim(ROOT_PATH, '/\\') . '/' . ltrim($rel, '/');
}

function public_url_from_rel(string $rel): string {
  return rtrim(BASE_URL, '/') . '/' . ltrim($rel, '/');
}

function fix_jpeg_orientation_if_needed($im, string $srcPath) {
  if (!function_exists('exif_read_data')) return $im;
  $exif = @exif_read_data($srcPath);
  $ori = (int)($exif['Orientation'] ?? 0);
  if ($ori === 3) return imagerotate($im, 180, 0);
  if ($ori === 6) return imagerotate($im, -90, 0);
  if ($ori === 8) return imagerotate($im, 90, 0);
  return $im;
}
function resize_if_tall($im, int $maxH = MAX_IMAGE_HEIGHT) {
  $w = imagesx($im);
  $h = imagesy($im);
  if ($h <= $maxH || $w <= 0) return $im;

  $ratio = $maxH / $h;
  $newW = max(1, (int)round($w * $ratio));
  $newH = $maxH;

  $dst = imagecreatetruecolor($newW, $newH);

  imagealphablending($dst, false);
  imagesavealpha($dst, true);
  $transparent = imagecolorallocatealpha($dst, 0, 0, 0, 127);
  imagefill($dst, 0, 0, $transparent);

  imagecopyresampled($dst, $im, 0, 0, 0, 0, $newW, $newH, $w, $h);
  imagedestroy($im);

  return $dst;
}

function convert_to_webp(string $srcPath, string $mime, string $destPath): bool {
  if (!function_exists('imagewebp')) return false;

  $im = null;

  switch ($mime) {
    case 'image/jpeg':
      if (!function_exists('imagecreatefromjpeg')) return false;
      $im = @imagecreatefromjpeg($srcPath);
      if (!$im) return false;
      $im2 = fix_jpeg_orientation_if_needed($im, $srcPath);
      if ($im2 !== $im) { imagedestroy($im); $im = $im2; }
      break;
      
       case 'image/gif':
      if (!function_exists('imagecreatefromgif')) return false;
      $im = @imagecreatefromgif($srcPath);
      if (!$im) return false;
      if (function_exists('imagepalettetotruecolor')) @imagepalettetotruecolor($im);
      @imagesavealpha($im, true);
      break;

    case 'image/png':
      if (!function_exists('imagecreatefrompng')) return false;
      $im = @imagecreatefrompng($srcPath);
      if (!$im) return false;
      if (function_exists('imagepalettetotruecolor')) @imagepalettetotruecolor($im);
      @imagesavealpha($im, true);
      break;

    case 'image/webp':
      if (!function_exists('imagecreatefromwebp')) return false;
      $im = @imagecreatefromwebp($srcPath);
      if (!$im) return false;
      @imagesavealpha($im, true);
      break;

    default:
      return false;
  }
 $im = resize_if_tall($im, MAX_IMAGE_HEIGHT);
  $quality = ($mime === 'image/png') ? 100 : 92;
  $ok = @imagewebp($im, $destPath, $quality);
  imagedestroy($im);

  return (bool)$ok;
}

function format_kb(int $bytes): string {
  $kb = $bytes / 1024;
  if ($kb < 1024) return number_format($kb, 1, '.', '') . ' KB';
  $mb = $kb / 1024;
  return number_format($mb, 1, '.', '') . ' MB';
}

function post_int(string $key, int $default = 0): int {
  $v = (int)($_POST[$key] ?? $default);
  return $v;
}

/* ---------------------------
   AJAX
----------------------------*/
if (is_post()) {
  if (!csrf_validate($_POST['csrf_token'] ?? null)) {
    json_out(['ok' => false, 'message' => 'Güvenlik doğrulaması başarısız.'], 403);
  }

  $action = (string)($_POST['action'] ?? '');

  try {
    switch ($action) {
      case 'ajax_upload': {
        if (!isset($_FILES['file'])) json_out(['ok' => false, 'message' => 'Dosya bulunamadı.'], 400);

        $f = $_FILES['file'];
        $err = (int)($f['error'] ?? UPLOAD_ERR_NO_FILE);
if ($err !== UPLOAD_ERR_OK) {
  $map = [
    UPLOAD_ERR_INI_SIZE   => 'Dosya sunucu limitini aşıyor (upload_max_filesize).',
    UPLOAD_ERR_FORM_SIZE  => 'Dosya form limitini aşıyor.',
    UPLOAD_ERR_PARTIAL    => 'Dosya kısmen yüklendi.',
    UPLOAD_ERR_NO_FILE    => 'Dosya seçilmedi.',
    UPLOAD_ERR_NO_TMP_DIR => 'Sunucuda geçici klasör yok (tmp).',
    UPLOAD_ERR_CANT_WRITE => 'Disk yazma hatası.',
    UPLOAD_ERR_EXTENSION  => 'Bir PHP eklentisi yüklemeyi durdurdu.',
  ];
  $msg = $map[$err] ?? 'Dosya yüklenemedi.';
  json_out(['ok' => false, 'message' => $msg, 'upload_err' => $err], 400);
}

        $tmp = (string)($f['tmp_name'] ?? '');
        if ($tmp === '' || !is_uploaded_file($tmp)) json_out(['ok' => false, 'message' => 'Geçersiz yükleme.'], 400);

        $sizeIn = (int)($f['size'] ?? 0);
        if ($sizeIn <= 0) json_out(['ok' => false, 'message' => 'Dosya boyutu hatalı.'], 400);
        if ($sizeIn > MAX_UPLOAD_BYTES) json_out(['ok' => false, 'message' => 'Maksimum dosya boyutu 10MB.'], 400);

        $finfo = new finfo(FILEINFO_MIME_TYPE);
        $mimeIn = (string)($finfo->file($tmp) ?: '');
        if (!allowed_mime($mimeIn)) json_out(['ok' => false, 'message' => 'Sadece jpg / png / gif / webp yükleyebilirsin.'], 400);


        if (!function_exists('imagewebp')) json_out(['ok' => false, 'message' => 'Sunucuda WebP dönüştürme aktif değil (GD imagewebp).'], 500);

        $sortKey = post_int('sort_key', 0);
        if ($sortKey <= 0) $sortKey = (int)(microtime(true) * 1000000);

        $destDir = uploads_abs_dir($tesisId);
        $destRelDir = uploads_rel_dir($tesisId);

        $finalName = '';
        $finalAbs = '';
        $finalRel = '';

        for ($i = 0; $i < 10; $i++) {
          $finalName = random_code(10) . '.webp';
          $finalAbs = rtrim($destDir, '/\\') . '/' . $finalName;
          if (!file_exists($finalAbs)) {
            $finalRel = $destRelDir . '/' . $finalName;
            break;
          }
        }
        if ($finalRel === '') json_out(['ok' => false, 'message' => 'Dosya adı üretilemedi.'], 500);

        $ok = convert_to_webp($tmp, $mimeIn, $finalAbs);
        if (!$ok) json_out(['ok' => false, 'message' => 'Görsel WebP formatına çevrilemedi.'], 500);

        $sizeOut = (int)@filesize($finalAbs);
        if ($sizeOut <= 0) $sizeOut = $sizeIn;

        $st = db()->prepare("
          INSERT INTO tesis_medya (tesis_id, dosya_adi, dosya_yol, mime, boyut, sort_key)
          VALUES (:tesis_id, :dosya_adi, :dosya_yol, :mime, :boyut, :sort_key)
        ");
        $st->execute([
          ':tesis_id'  => $tesisId,
          ':dosya_adi' => $finalName,
          ':dosya_yol' => $finalRel,
          ':mime'      => 'image/webp',
          ':boyut'     => $sizeOut,
          ':sort_key'  => $sortKey,
        ]);

        $id = (int)db()->lastInsertId();

        json_out([
          'ok' => true,
          'id' => $id,
          'file' => $finalName,
          'rel' => $finalRel,
          'url' => public_url_from_rel($finalRel),
          'size' => $sizeOut,
          'size_kb' => format_kb($sizeOut),
        ]);
      }

      case 'ajax_delete': {
        $id = post_int('id', 0);
        if ($id <= 0) json_out(['ok' => false, 'message' => 'Geçersiz istek.'], 400);

        $st = db()->prepare("SELECT id, tesis_id, dosya_yol FROM tesis_medya WHERE id = :id LIMIT 1");
        $st->execute([':id' => $id]);
        $row = $st->fetch();

        if (!$row) json_out(['ok' => false, 'message' => 'Kayıt bulunamadı.'], 404);
        if ((int)$row['tesis_id'] !== $tesisId && !$isAdmin) json_out(['ok' => false, 'message' => 'Bu görseli silemezsin.'], 403);

        $path = abs_path_from_rel((string)$row['dosya_yol']);

        $st2 = db()->prepare("DELETE FROM tesis_medya WHERE id = :id");
        $st2->execute([':id' => $id]);

        if (is_file($path)) @unlink($path);

        json_out(['ok' => true, 'deleted_ids' => [$id]]);
      }

      case 'ajax_bulk_delete': {
        $ids = $_POST['ids'] ?? [];
        if (!is_array($ids)) $ids = [];
        $ids = array_values(array_unique(array_map('intval', $ids)));
        $ids = array_values(array_filter($ids, fn($x) => $x > 0));

        if (!$ids) json_out(['ok' => false, 'message' => 'Seçim yok.'], 400);
        if (count($ids) > MAX_BULK_DELETE) json_out(['ok' => false, 'message' => 'Çok fazla seçim.'], 400);

        $placeholders = implode(',', array_fill(0, count($ids), '?'));
        $st = db()->prepare("SELECT id, tesis_id, dosya_yol FROM tesis_medya WHERE id IN ($placeholders)");
        $st->execute($ids);
        $rows = $st->fetchAll();

        if (!$rows) json_out(['ok' => false, 'message' => 'Kayıt bulunamadı.'], 404);

        $allowed = [];
        $paths = [];

        foreach ($rows as $r) {
          $rid = (int)$r['id'];
          $rt  = (int)$r['tesis_id'];
          if ($isAdmin || $rt === $tesisId) {
            $allowed[] = $rid;
            $paths[$rid] = abs_path_from_rel((string)$r['dosya_yol']);
          }
        }

        if (!$allowed) json_out(['ok' => false, 'message' => 'Silme yetkin yok.'], 403);

        $ph2 = implode(',', array_fill(0, count($allowed), '?'));
        $del = db()->prepare("DELETE FROM tesis_medya WHERE id IN ($ph2)");
        $del->execute($allowed);

        foreach ($allowed as $rid) {
          $p = $paths[$rid] ?? null;
          if ($p && is_file($p)) @unlink($p);
        }

        json_out(['ok' => true, 'deleted_ids' => $allowed, 'deleted_count' => count($allowed)]);
      }

      default:
        json_out(['ok' => false, 'message' => 'Geçersiz işlem.'], 400);
    }
  } catch (Throwable $e) {
    error_log('ortam-kutuphanesi hata: ' . $e->getMessage());
    json_out(['ok' => false, 'message' => 'Sunucu hatası.'], 500);
  }
}

/* ---------------------------
   PAGE
----------------------------*/
$limit  = 60;
$page   = max(1, (int)($_GET['p'] ?? 1));
$offset = ($page - 1) * $limit;

$totalSt = db()->prepare("SELECT COUNT(*) FROM tesis_medya WHERE tesis_id = :tesis_id");
$totalSt->execute([':tesis_id' => $tesisId]);
$total = (int)$totalSt->fetchColumn();

$listSt = db()->prepare("
  SELECT id, dosya_adi, dosya_yol, boyut
  FROM tesis_medya
  WHERE tesis_id = :tesis_id
  ORDER BY sort_key DESC, id DESC
  LIMIT {$limit} OFFSET {$offset}
");
$listSt->execute([':tesis_id' => $tesisId]);
$items = $listSt->fetchAll();

$totalPages = max(1, (int)ceil($total / $limit));
$csrf = csrf_token();
?>
<!doctype html>
<html lang="tr">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <title>Ortam Kütüphanesi</title>
</head>
<body class="panel-body">


  <div class="panel-shell">
    <?php include __DIR__ . '/sidebar.php'; ?>

    <div class="panel-main">
      <header class="topbar">

        <div class="topbar-title">
          <div class="topbar-h1">Ortam Kütüphanesi</div>
          <div class="topbar-sub muted">Sürükle-bırak veya görsel seçerek yükle.</div>
        </div>

        <div class="topbar-right">
          <div class="pill">
            <span class="muted">Toplam</span>
            <span class="pill-strong" id="mediaTotal"><?= (int)$total ?></span>
          </div>
        </div>
      </header>

      <main class="content">

        <section class="panel-section media-section" style="margin-top:0;">
          <div id="dropZone" class="upload-zone">
            <div class="upload-zone-title">Görselleri buraya sürükle bırak</div>
            <div class="upload-zone-sub">ya da aşağıdan seç</div>
            <div class="upload-zone-actions">
              <label class="file-btn" for="fileInput">Görsel Seç</label>
              <input id="fileInput" class="file-input" type="file" accept="image/jpeg,image/png,image/gif,image/webp" multiple>
              <div class="media-hint">Maks: 10 görsel • 10MB • jpg/png/webp • otomatik WebP</div>
            </div>
          </div>
        </section>

        <section class="panel-section">
          <div class="section-head">
            <div class="section-title">Görseller</div>
            <div class="section-right">
              <div class="muted section-page"><?= $page ?> / <?= $totalPages ?></div>
              <button id="bulkToggle" class="btn btn-ghost btn-sm" type="button">Toplu Seçim</button>
            </div>
          </div>

          <div id="bulkBar" class="bulk-bar is-hidden">
            <div class="bulk-left"><span class="bulk-count" id="bulkCount">0</span><span class="muted">seçildi</span></div>
            <div class="bulk-right">
              <button id="bulkCancel" class="btn btn-ghost btn-sm" type="button">Vazgeç</button>
              <button id="bulkDelete" class="btn btn-danger btn-sm" type="button" disabled>Kalıcı Olarak Sil</button>
            </div>
          </div>

          <div id="mediaGrid" class="media-grid">
            <?php foreach ($items as $it): ?>
              <?php
                $url = public_url_from_rel((string)$it['dosya_yol']);
                $file = (string)$it['dosya_adi'];
                $kb = format_kb((int)$it['boyut']);
              ?>
              <div class="media-card" data-id="<?= (int)$it['id'] ?>" data-file="<?= e($file) ?>" data-src="<?= e($url) ?>">
                <button class="media-thumb js-preview" type="button" aria-label="Önizle">
                  <img class="media-img" src="<?= e($url) ?>" alt="">
                </button>
                <div class="media-meta">
                  <div class="media-name" title="<?= e($file) ?>"><?= e($file) ?></div>
                  <div class="media-sub muted"><?= e($kb) ?></div>
                </div>
                <div class="media-actions">
                  <button class="icon-btn js-delete" type="button" title="Sil">🗑️</button>
                </div>
              </div>
            <?php endforeach; ?>
          </div>

          <?php if ($totalPages > 1): ?>
            <div class="media-pager">
              <?php if ($page > 1): ?><a class="btn btn-ghost btn-sm" href="?p=<?= $page - 1 ?>">← Önceki</a><?php endif; ?>
              <?php if ($page < $totalPages): ?><a class="btn btn-ghost btn-sm" href="?p=<?= $page + 1 ?>">Sonraki →</a><?php endif; ?>
            </div>
          <?php endif; ?>

        </section>

      </main>
    </div>
  </div>

  <script>
    window.BS_MEDIA = { endpoint:'ortam-kutuphanesi.php', csrf:'<?= e($csrf) ?>' };
  </script>
  <script src="upload.js"></script>
</body>
</html>
