// /home/bungalowsepeti/public_html/panel/js/popup.js
(() => {
  "use strict";

  const qs = (s, r = document) => r.querySelector(s);
  const qsa = (s, r = document) => Array.from(r.querySelectorAll(s));

  const BS = (window.BS = window.BS || {});

  // toast.js varsa BS.toast kullanır
  const toastDefault = (msg, type = "info", timeout) => {
    if (typeof BS.toast === "function") return BS.toast(msg, type, timeout);
    // en basit fallback
    try { console.log("[toast]", type, msg); } catch {}
  };

  const escapeHtml = (str) =>
    String(str).replace(/[&<>"']/g, (m) => ({ "&": "&amp;", "<": "&lt;", ">": "&gt;", '"': "&quot;", "'": "&#39;" }[m]));

  const formatKB = (bytes) => {
    const kb = (Number(bytes) || 0) / 1024;
    if (kb < 1024) return `${kb.toFixed(1)} KB`;
    const mb = kb / 1024;
    return `${mb.toFixed(1)} MB`;
  };

  const toIntId = (x) => {
    if (Number.isInteger(x) && x > 0) return x;
    if (typeof x === "number" && Number.isFinite(x)) {
      const n = Math.trunc(x);
      return n > 0 ? n : 0;
    }
    if (typeof x === "string") {
      const t = x.trim();
      if (!/^\d+$/.test(t)) return 0;
      const n = parseInt(t, 10);
      return Number.isInteger(n) && n > 0 ? n : 0;
    }
    return 0;
  };

  const uniqIntIds = (arr) => {
    const out = [];
    const seen = new Set();
    (arr || []).forEach((x) => {
      const id = toIntId(x);
      if (!id) return;
      if (seen.has(id)) return;
      seen.add(id);
      out.push(id);
    });
    return out;
  };

  /* ---------------------------
     Mini modal helper
  ----------------------------*/
  BS.openMiniModal = (html, opts = {}) => {
    const overlay = document.createElement("div");
    overlay.className = "mini-modal-overlay";
    overlay.innerHTML = html;
    document.body.appendChild(overlay);

    const onClose = typeof opts.onClose === "function" ? opts.onClose : null;

    const close = (reason = "user") => {
      if (overlay && overlay.parentNode) overlay.remove();
      document.removeEventListener("keydown", onKeyDown, true);
      if (reason === "user" && onClose) onClose();
    };

    const onKeyDown = (e) => {
      if (e.key === "Escape") close("user");
    };

    document.addEventListener("keydown", onKeyDown, true);

    overlay.addEventListener("click", (e) => {
      if (e.target === overlay) close("user");
    });

    qs(".mini-modal-x", overlay)?.addEventListener("click", () => close("user"));

    return { overlay, close };
  };

  /* ---------------------------
     Confirm modal (Pasif / Arşiv teyidi için)
  ----------------------------*/
  BS.confirmModal = (opts = {}) => {
    const title = escapeHtml(opts.title || "Onay");
    const text = escapeHtml(opts.text || "");
    const okText = escapeHtml(opts.okText || "Evet");
    const cancelText = escapeHtml(opts.cancelText || "Vazgeç");

    return new Promise((resolve) => {
      let settled = false;

      const done = (val) => {
        if (settled) return;
        settled = true;
        close("done");
        resolve(!!val);
      };

      const cancel = () => done(false);

      const { overlay, close } = BS.openMiniModal(
        `
        <div class="mini-modal" role="dialog" aria-modal="true" style="max-width:520px;">
          <div class="mini-modal-hd">
            <div class="mini-modal-title">${title}</div>
            <button class="mini-modal-x" type="button" aria-label="Kapat">✕</button>
          </div>

          <div class="mini-modal-bd" style="padding:14px 14px 6px 14px;">
            <div style="font-size:14px;line-height:1.5;color:#0b1635;">${text}</div>
          </div>

          <div class="mini-modal-ft" style="display:flex;justify-content:flex-end;gap:10px;padding:12px 14px 14px 14px;">
            <button class="btn btn-ghost btn-sm" type="button" id="cmCancel">${cancelText}</button>
            <button class="btn btn-primary btn-sm" type="button" id="cmOk">${okText}</button>
          </div>
        </div>
        `,
        { onClose: cancel }
      );

      qs("#cmCancel", overlay)?.addEventListener("click", cancel);
      qs("#cmOk", overlay)?.addEventListener("click", () => done(true));

      overlay.addEventListener("keydown", (e) => {
        if (e.key === "Enter") done(true);
      });

      qs("#cmOk", overlay)?.focus?.();
    });
  };

  /* ---------------------------
     Media picker modal (Kapak / Galeri)
  ----------------------------*/
  BS.openMediaPicker = async (opts = {}) => {
    const mode = opts.mode === "cover" ? "cover" : "gallery";

    const fetchItems = opts.fetchItems;
    const xhrPost = opts.xhrPost;

    const csrf = String(opts.csrf || "");
    const uploadEndpoint = String(opts.uploadEndpoint || "");
    const maxGallery = Number(opts.maxGallery || 10) || 10;

    const toast = typeof opts.toast === "function" ? opts.toast : toastDefault;

    const coverId = toIntId(opts.coverId);
    const pickedIds = Array.isArray(opts.pickedIds) ? opts.pickedIds : [];

    if (typeof fetchItems !== "function") throw "fetchItems eksik.";
    if (typeof xhrPost !== "function") throw "xhrPost eksik.";
    if (!csrf || !uploadEndpoint) throw "CSRF / upload endpoint eksik.";

    let items = [];
    try {
      items = await fetchItems();
    } catch (err) {
      throw err || "Liste alınamadı.";
    }

    return new Promise((resolve) => {
      let settled = false;

      const picked = new Set(uniqIntIds(pickedIds).filter((id) => id > 0));
      if (mode === "cover") {
        const first = picked.values().next().value || 0;
        picked.clear();
        if (first) picked.add(first);
      } else {
        if (coverId) picked.delete(coverId);
        if (picked.size > maxGallery) {
          const arr = Array.from(picked).slice(0, maxGallery);
          picked.clear();
          arr.forEach((x) => picked.add(x));
        }
      }

      const done = (val) => {
        if (settled) return;
        settled = true;
        close("done");
        resolve(val);
      };

      const cancel = () => {
        if (settled) return;
        settled = true;
        resolve(null);
      };

      const { overlay, close } = BS.openMiniModal(
        `
        <div class="mini-modal" role="dialog" aria-modal="true" style="max-width:920px;">
          <div class="mini-modal-hd">
            <div class="mini-modal-title">${mode === "cover" ? "Kapak Seç" : "Galeri Seç"}</div>
            <button class="mini-modal-x" type="button" aria-label="Kapat">✕</button>
          </div>

          <div class="mini-modal-bd" style="padding:12px;">
            <div style="display:flex;gap:10px;align-items:center;justify-content:space-between;flex-wrap:wrap;margin-bottom:10px;">
              <div class="muted" style="font-size:14px;">Ortam Kütüphanesi</div>

              <div style="display:flex;gap:8px;align-items:center;flex-wrap:wrap;">
                <label class="btn btn-ghost btn-sm" style="cursor:pointer;position:relative;">
                  Yeni Görsel Yükle
                  <input
                    id="pickUpload"
                    type="file"
                    accept="image/jpeg,image/png,image/webp"
                    ${mode === "cover" ? "" : "multiple"}
                    style="position:absolute;left:-9999px;top:-9999px;"
                  >
                </label>

                ${mode === "gallery" ? `<button class="btn btn-primary btn-sm" type="button" id="pickOk">Seçilenleri Ekle</button>` : ``}
              </div>
            </div>

            <div class="media-grid" id="pickGrid" style="grid-template-columns:repeat(6,minmax(0,1fr));gap:10px;"></div>

            ${mode === "gallery" ? `<div class="muted" style="font-size:12px;margin-top:10px;">Max ${maxGallery} görsel seçebilirsin.</div>` : ``}
          </div>
        </div>
        `,
        { onClose: cancel }
      );

      const grid = qs("#pickGrid", overlay);
      const okBtn = qs("#pickOk", overlay);
      const upInp = qs("#pickUpload", overlay);

      const makePh = (name) => {
        const card = document.createElement("div");
        card.className = "media-card is-loading";
        card.innerHTML = `
          <div class="media-thumb media-thumb-loading">
            <div class="media-skeleton"></div>
            <div class="media-progress"><div class="media-progress-bar" style="width:0%"></div></div>
            <div class="media-wait">Kaydediliyor...</div>
          </div>
          <div class="media-meta">
            <div class="media-name" title="${escapeHtml(name)}">${escapeHtml(name)}</div>
            <div class="media-sub muted">0.0 KB</div>
          </div>
        `;
        return card;
      };

      const draw = () => {
        if (!grid) return;
        grid.innerHTML = "";

        items.forEach((it) => {
          const id = toIntId(it.id);
          if (!id) return;

          const isSelected = picked.has(id);

          const card = document.createElement("div");
          card.className = "media-card" + (isSelected ? " is-selected" : "");
          card.dataset.id = String(id);
          card.dataset.src = String(it.url || "");

          card.innerHTML = `
            <button class="media-thumb js-pick" type="button" aria-label="Seç">
              <img class="media-img" src="${escapeHtml(it.url)}" alt="">
            </button>
            <div class="media-meta">
              <div class="media-name" title="${escapeHtml(it.file)}">${escapeHtml(it.file)}</div>
              <div class="media-sub muted">${escapeHtml(formatKB(it.size || 0))}</div>
            </div>
          `;

          grid.appendChild(card);
        });
      };

      draw();

      grid?.addEventListener(
        "click",
        (e) => {
          const pickBtn = e.target.closest(".js-pick");
          if (!pickBtn) return;

          e.preventDefault();
          e.stopPropagation();

          const card = pickBtn.closest(".media-card");
          const id = toIntId(card?.dataset?.id);
          if (!id) return;

          if (mode === "cover") {
            done({ coverId: id });
            return;
          }

          if (coverId && id === coverId) {
            toast("Kapak görseli galeriye eklenemez.", "error");
            return;
          }

          if (picked.has(id)) {
            picked.delete(id);
            card.classList.remove("is-selected");
          } else {
            if (picked.size >= maxGallery) {
              toast(`Max ${maxGallery} görsel.`, "error");
              return;
            }
            picked.add(id);
            card.classList.add("is-selected");
          }
        },
        { capture: true }
      );

      if (okBtn) {
        okBtn.addEventListener("click", () => {
          const arr = uniqIntIds(Array.from(picked));
          done({ galleryIds: arr });
        });
      }

      upInp?.addEventListener("change", () => {
        const files = Array.from(upInp.files || []).filter(Boolean);
        upInp.value = "";
        if (!files.length) return;

        const max = mode === "cover" ? 1 : Math.min(maxGallery, files.length);
        const list = files.slice(0, max);
        const base = Date.now() * 1000;

        list.forEach((file, i) => {
          if (file.size > 10 * 1024 * 1024) {
            toast(`${file.name}: 10MB sınırını aşıyor.`, "error");
            return;
          }

          const ph = makePh(file.name);
          grid?.prepend(ph);

          const bar = qs(".media-progress-bar", ph);
          const sub = qs(".media-sub", ph);

          const fd = new FormData();
          fd.append("csrf_token", csrf);
          fd.append("action", "ajax_upload");
          fd.append("sort_key", String(base + i));
          fd.append("file", file);

          xhrPost(uploadEndpoint, fd, (p) => {
            if (bar) bar.style.width = `${p}%`;
            if (sub) sub.textContent = formatKB(file.size * (p / 100));
          })
            .then((data) => {
              if (!data || !data.ok) throw data?.message || "Yükleme başarısız.";

              const newId = toIntId(data?.id);
              if (!newId) throw "Yükleme sonrası görsel id okunamadı.";

              const newItem = {
                id: newId,
                file: String(data.file || file.name || ""),
                url: String(data.url || ""),
                size: Number(data.size || file.size || 0),
              };

              items.unshift(newItem);
              draw();

              if (mode === "cover") {
                toast("Yüklendi.", "success", 2200);
                done({ coverId: newId });
                return;
              }

              if (picked.size < maxGallery && newId !== coverId) picked.add(newId);
              toast("Yüklendi.", "success", 2200);

              qsa(`.media-card[data-id="${newId}"]`, grid).forEach((c) => c.classList.toggle("is-selected", picked.has(newId)));
              ph.remove();
            })
            .catch((err) => {
              ph.remove();
              toast(String(err || "Yükleme başarısız."), "error");
            });
        });
      });

      (okBtn || qs(".mini-modal-x", overlay))?.focus?.();
    });
  };
})();
